<?xml version="1.0" encoding="UTF-8"?>

<!--******************************
       DAISY XSL TRANSFORM (ja)

    Make an XSL capable browser
    understand DAISY markup.
****************************** -->

<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
xmlns:dtb="http://www.daisy.org/z3986/2005/dtbook/">

  <xsl:output method="html" indent="no"/>

  <!--******************************
   DOCBOOK, HEAD, META, LINK, BOOK
  *******************************-->

  <!-- docbook translates to html -->
  <xsl:template match="dtb:dtbook">
    <html>
      <xsl:apply-templates/>
    </html>
  </xsl:template>

  <!-- head maps directly -->
  <xsl:template match="dtb:head">
    <xsl:element name="head">
      <title>
        <xsl:value-of select="/dtb:dtbook/dtb:head/dtb:meta/@content"/>
      </title>
      <xsl:apply-templates/>
      <style type="text/css">
        span[id] {white-space:pre-wrap; padding-top:0.5em;}
        p {margin: 0;}
        div + p {margin-top: 0.5em;}
        span[id], div[style]>p {line-height: 2.6;}
      </style>
    </xsl:element>
  </xsl:template>

  <!-- meta maps directly
       Include: content
       If applicable, include: http-equiv, name
       NOTE: meta contains no content so no apply-templates necessary -->
  <xsl:template match="dtb:meta">
    <xsl:element name="meta">
      <xsl:if test="@http-equiv">
        <xsl:attribute name="http-equiv">
          <xsl:value-of select="@http-equiv"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@name">
        <xsl:attribute name="name">
          <xsl:value-of select="@name"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:attribute name="content">
        <xsl:value-of select="@content"/>
      </xsl:attribute>
    </xsl:element>
  </xsl:template>

  <!-- delete indent -->
  <xsl:template match="text()" name="delIndent">
    <xsl:param name="bodyText" select="."/>
    <xsl:choose>
      <xsl:when test="not(contains($bodyText, '&#xA;'))">
        <xsl:copy-of select="$bodyText"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="substring-before($bodyText, '&#xA;')"/>
        <xsl:call-template name="delIndent">
          <xsl:with-param name="bodyText" select= "substring-after($bodyText, '.')"/>
        </xsl:call-template>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <!-- book should be translated to body -->
  <xsl:template match="dtb:book">
    <body>
      <xsl:apply-templates/>
    </body>
  </xsl:template>


  <!--*******************************
   BODYMATTER
  ******************************* -->

  <xsl:template match="dtb:bodymatter">
    <div class="bodymatter">
      <xsl:apply-templates/>
    </div>
  </xsl:template>


  <!--***********************
             LEVELS
  ************************-->

  <!-- Levels map to div with class -->
  <xsl:template match="dtb:level | dtb:level1 | dtb:level2 | dtb:level3 | dtb:level4 | dtb:level5 | dtb:level6">
    <xsl:element name="div">
      <xsl:attribute name="class">
        <xsl:value-of select="local-name(.)" />
      </xsl:attribute>
      <xsl:apply-templates />
    </xsl:element>
  </xsl:template>

  <!--***********************
            HEADINGS
  ************************-->

  <!--h1...h6 map directly -->
  <xsl:template match="dtb:h1 | dtb:h2 | dtb:h3 | dtb:h4 | dtb:h5 | dtb:h6">
    <xsl:element name="{local-name(.)}">
      <xsl:apply-templates />
    </xsl:element>
  </xsl:template>

  <!--*************************
         PAGENUM, LINENUM
  ************************-->

  <!--Put the pagenum into a paragraph element if the parent is level or level1...level6 otherwise put it into a span
      Use the pagenum class for formatting -->
  <xsl:template match="dtb:pagenum">
        <xsl:element name="span">
          <xsl:attribute name="class">
            <xsl:value-of select="local-name(.)" />
          </xsl:attribute>
          <xsl:if test="@id">
            <xsl:attribute name="id">
              <xsl:value-of select="@id" />
            </xsl:attribute>
          </xsl:if>
          <xsl:if test="@smilref">
            <xsl:attribute name="smilref">
              <xsl:value-of select="@smilref" />
            </xsl:attribute>
          </xsl:if>
          <xsl:apply-templates />
        </xsl:element>
  </xsl:template>

  <!-- linenum is translated to span with class -->
  <xsl:template match="dtb:linenum">
    <xsl:element name="span">
      <xsl:attribute name="class">
        <xsl:value-of select="local-name(.)" />
      </xsl:attribute>
      <xsl:if test="@id">
        <xsl:attribute name="id">
          <xsl:value-of select="@id" />
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@smilref">
        <xsl:attribute name="smilref">
          <xsl:value-of select="@smilref" />
        </xsl:attribute>
      </xsl:if>
      <xsl:apply-templates />
    </xsl:element>
  </xsl:template>


  <!--*************************
        GENERAL BLOCKS
  ************************-->

  <!-- p maps directly -->
  <xsl:template match="dtb:p">
    <xsl:choose>
      <!--<hr/> -->
      <xsl:when test="@class='hr'">
        <hr/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:element name="p">
          <xsl:if test="@id">
            <xsl:attribute name="id">
              <xsl:value-of select="@id" />
            </xsl:attribute>
          </xsl:if>
	      <xsl:if test="@class">
	        <xsl:attribute name="style">
	          <xsl:value-of select="@class"/>
	        </xsl:attribute>
	      </xsl:if>
          <xsl:apply-templates />
        </xsl:element>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <!-- div maps directly -->
  <xsl:template match="dtb:div">
    <xsl:element name="div">
      <xsl:if test="@class">
        <xsl:attribute name="style">
          <xsl:value-of select="@class"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:apply-templates />
    </xsl:element>
  </xsl:template>

  <!-- prodnote maps to div with class
       prodnote may contain bear text or block level elements, thus it must be wrapped in something, but it can't be p, otherwise we may end up with nested p
       Exclude: render attribute, no way to express -->
  <xsl:template match="dtb:prodnote">
    <xsl:element name="span">
      <xsl:attribute name="class">
        <xsl:value-of select="local-name(.)" />
      </xsl:attribute>
      <xsl:if test="@id">
        <xsl:attribute name="id">
          <xsl:value-of select="@id" />
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@smilref">
        <xsl:attribute name="smilref">
          <xsl:value-of select="@smilref" />
        </xsl:attribute>
      </xsl:if>
      <xsl:apply-templates />
    </xsl:element>
  </xsl:template>

  <!-- sidebar maps to div with class
       Exclude: render attribute, no way to express -->
  <xsl:template match="dtb:sidebar">
    <xsl:element name="span">
      <xsl:attribute name="class">
        <xsl:value-of select="local-name(.)" />
      </xsl:attribute>
      <xsl:if test="@id">
        <xsl:attribute name="id">
          <xsl:value-of select="@id" />
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@smilref">
        <xsl:attribute name="smilref">
          <xsl:value-of select="@smilref" />
        </xsl:attribute>
      </xsl:if>
      <xsl:apply-templates />
    </xsl:element>
  </xsl:template>

  <!-- note maps to div with class -->
  <xsl:template match="dtb:note">
    <xsl:element name="span">
      <xsl:attribute name="class">
        <xsl:value-of select="local-name(.)" />
      </xsl:attribute>
      <xsl:if test="@id">
        <xsl:attribute name="id">
          <xsl:value-of select="@id" />
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@smilref">
        <xsl:attribute name="smilref">
          <xsl:value-of select="@smilref" />
        </xsl:attribute>
      </xsl:if>
      <xsl:for-each select="*">
        <xsl:choose>
          <xsl:when test="name(.)='p'">
            <xsl:element name="span">
              <xsl:apply-templates/>
            </xsl:element>
          </xsl:when>
        </xsl:choose>
      </xsl:for-each>
    </xsl:element>
  </xsl:template>

  <!-- annotation maps to div with class -->
  <xsl:template match="dtb:annotation">
    <xsl:element name="span">
      <xsl:attribute name="class">
        <xsl:value-of select="local-name(.)" />
      </xsl:attribute>
      <xsl:if test="@id">
        <xsl:attribute name="id">
          <xsl:value-of select="@id" />
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@smilref">
        <xsl:attribute name="smilref">
          <xsl:value-of select="@smilref" />
        </xsl:attribute>
      </xsl:if>
      <xsl:for-each select="*">
        <xsl:choose>
          <xsl:when test="name(.)='p'">
            <xsl:element name="span">
              <xsl:apply-templates/>
            </xsl:element>
          </xsl:when>
        </xsl:choose>
      </xsl:for-each>
    </xsl:element>
  </xsl:template>


  <!-- Map a line to a br tag -->
  <xsl:template match="dtb:line">
    <xsl:element name="p">
      <xsl:attribute name="class">
        <xsl:value-of select="local-name(.)" />
      </xsl:attribute>
      <xsl:apply-templates />
    </xsl:element>
  </xsl:template>
  <!--*************************
       GENERAL INLINES
  ************************-->

  <!-- em maps directly -->
  <xsl:template match="dtb:em">
    <i>
      <xsl:apply-templates/>
    </i>
  </xsl:template>

  <!-- strong maps directly -->
  <xsl:template match="dtb:strong">
    <b>
      <xsl:apply-templates/>
    </b>
  </xsl:template>

  <!-- span maps to span for classes underline, small-caps, ruby -->
  <xsl:template match="dtb:span">
    <xsl:choose>
      <xsl:when test="@class='underline'">
          <span style="border-bottom:solid 1px currentColor;">
          <xsl:apply-templates/>
        </span>
      </xsl:when>
      <xsl:when test="@class='ruby'">
        <ruby>
          <xsl:apply-templates/>
        </ruby>
      </xsl:when>
      <xsl:when test="@class='ruby original'">
        <ruby class="original">
          <xsl:apply-templates/>
        </ruby>
      </xsl:when>
      <xsl:when test="@class='rb'">
        <rb>
          <xsl:apply-templates/>
        </rb>
      </xsl:when>
      <xsl:when test="@class='rt'">
        <rt>
          <xsl:apply-templates/>
        </rt>
      </xsl:when>
      <xsl:when test="@class='rp'">
        <rp>
          <xsl:apply-templates/>
        </rp>
      </xsl:when>
      <xsl:otherwise>
        <xsl:element name="span">
          <xsl:if test="@class">
            <xsl:attribute name="style">
              <xsl:value-of select="@class" />
            </xsl:attribute>
          </xsl:if>
          <xsl:apply-templates />
        </xsl:element>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <!-- sub maps directly-->
  <xsl:template match="dtb:sub">
    <sub>
      <xsl:apply-templates/>
    </sub>
  </xsl:template>

  <!--sup maps directly-->
  <xsl:template match="dtb:sup">
    <sup>
      <xsl:apply-templates/>
    </sup>
  </xsl:template>


  <!-- br maps directly
       NOTE: no apply-templates needed since this tag is always self closing-->
  <xsl:template match="dtb:br">
    <xsl:element name="br">
      <xsl:if test="@class">
        <xsl:attribute name="clear">
          <xsl:value-of select="@class"/>
        </xsl:attribute>
      </xsl:if>
    </xsl:element>
  </xsl:template>

  <!-- noteref maps to span with class -->
  <xsl:template match="dtb:noteref">
    <xsl:element name="span">
      <xsl:attribute name="class">
        <xsl:value-of select="local-name(.)" />
      </xsl:attribute>
      <xsl:if test="@id">
        <xsl:attribute name="id">
          <xsl:value-of select="@id" />
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@smilref">
        <xsl:attribute name="smilref">
          <xsl:value-of select="@smilref" />
        </xsl:attribute>
      </xsl:if>
      <xsl:apply-templates />
    </xsl:element>
  </xsl:template>

  <!-- annoref maps to span with class -->
  <xsl:template match="dtb:annoref">
    <xsl:element name="span">
      <xsl:attribute name="class">
        <xsl:value-of select="local-name(.)" />
      </xsl:attribute>
      <xsl:if test="@id">
        <xsl:attribute name="id">
          <xsl:value-of select="@id" />
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@smilref">
        <xsl:attribute name="smilref">
          <xsl:value-of select="@smilref" />
        </xsl:attribute>
      </xsl:if>
      <xsl:apply-templates />
    </xsl:element>
  </xsl:template>

  <!-- no equivalent tag -->
  <xsl:template match="dtb:sent">
    <xsl:element name="span">
      <xsl:if test="@id">
        <xsl:attribute name="id">
          <xsl:value-of select="@id" />
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@smilref">
        <xsl:attribute name="smilref">
          <xsl:value-of select="@smilref" />
        </xsl:attribute>
      </xsl:if>
      <xsl:apply-templates />
    </xsl:element>
  </xsl:template>

  <!--*************************
           LISTS
  ************************-->

  <!--Get fancy with the various list types-->

  <!-- An unordered list will be wrapped in ul tags -->
  <xsl:template match="dtb:list[@type='ul']">
    <ul>
      <xsl:apply-templates/>
    </ul>
  </xsl:template>


  <!-- li maps directly -->
  <xsl:template match="dtb:li">
    <li>
      <xsl:apply-templates/>
    </li>
  </xsl:template>

  <!--*************************
            TABLES
  ************************ *** -->

  <xsl:template match="dtb:table">
    <xsl:element name="table">
      <xsl:if test="@border">
        <xsl:attribute name="border">
          <xsl:value-of select="@border"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@cellpadding">
        <xsl:attribute name="cellpadding">
          <xsl:value-of select="@cellpadding"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:apply-templates/>
    </xsl:element>
  </xsl:template>

  <!-- tr maps directly
       If applicable, include: align, char, charoff, valign -->
  <xsl:template match="dtb:tr">
    <xsl:element name="tr">
      <xsl:apply-templates/>
    </xsl:element>
  </xsl:template>

  <!-- td maps directly
       If applicable, include: align, axis, char, charoff, colspan, headers, rowspan, scope, span, valign -->
  <xsl:template match="dtb:td">
    <xsl:element name="td">
      <xsl:if test="@colspan">
        <xsl:attribute name="colspan">
          <xsl:value-of select="@colspan"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@rowspan">
        <xsl:attribute name="rowspan">
          <xsl:value-of select="@rowspan"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@span">
        <xsl:attribute name="span">
          <xsl:value-of select="@span"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@class">
        <xsl:attribute name="style">
          <xsl:value-of select="@class"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:apply-templates/>
    </xsl:element>
  </xsl:template>

  <!--*************************
            IMAGES
  ************************ *** -->

  <!-- img maps directly
       Include: alt, src
       If applicable, include: longdesc, height, width
       NOTE: img is self closing so no apply-templates necessary -->
  <xsl:template match="dtb:img">
    <xsl:element name="img">
      <xsl:attribute name="alt">
        <xsl:value-of select="@alt"/>
      </xsl:attribute>
      <xsl:attribute name="src">
        <xsl:value-of select="@src"/>
      </xsl:attribute>
      <xsl:attribute name="id">
        <xsl:value-of select="@id"/>
      </xsl:attribute>
      <xsl:attribute name="style">padding: 3px</xsl:attribute>
      <xsl:if test="@height">
        <xsl:attribute name="height">
          <xsl:value-of select="@height"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@width">
        <xsl:attribute name="width">
          <xsl:value-of select="@width"/>
        </xsl:attribute>
      </xsl:if>
    </xsl:element>
  </xsl:template>
</xsl:stylesheet>
